-- Create Hotels Configuration Table
-- This table stores all hotels connected to the SmartHotel SaaS platform

CREATE TABLE IF NOT EXISTS hotels (
    id UUID DEFAULT gen_random_uuid() PRIMARY KEY,
    hotel_id VARCHAR(50) UNIQUE NOT NULL,
    name VARCHAR(255) NOT NULL,
    status VARCHAR(20) NOT NULL DEFAULT 'pending' CHECK (status IN ('active', 'inactive', 'pending')),
    ai_configured BOOLEAN DEFAULT FALSE,
    
    -- Contact Information
    website VARCHAR(255),
    email VARCHAR(255),
    phone VARCHAR(50),
    address TEXT,
    
    -- Configuration
    supabase_url VARCHAR(255),
    supabase_key_configured BOOLEAN DEFAULT FALSE,
    openai_key_configured BOOLEAN DEFAULT FALSE,
    
    -- Hotel Details
    rooms INTEGER,
    description TEXT,
    
    -- Subscription & Billing
    subscription_plan VARCHAR(50) DEFAULT 'free',
    monthly_revenue DECIMAL(10, 2) DEFAULT 0,
    
    -- Timestamps
    joined_date TIMESTAMP WITH TIME ZONE DEFAULT TIMEZONE('utc'::text, NOW()) NOT NULL,
    last_active TIMESTAMP WITH TIME ZONE DEFAULT TIMEZONE('utc'::text, NOW()) NOT NULL,
    created_at TIMESTAMP WITH TIME ZONE DEFAULT TIMEZONE('utc'::text, NOW()) NOT NULL,
    updated_at TIMESTAMP WITH TIME ZONE DEFAULT TIMEZONE('utc'::text, NOW()) NOT NULL
);

-- Create indexes for better query performance
CREATE INDEX IF NOT EXISTS idx_hotels_hotel_id ON hotels(hotel_id);
CREATE INDEX IF NOT EXISTS idx_hotels_status ON hotels(status);
CREATE INDEX IF NOT EXISTS idx_hotels_joined_date ON hotels(joined_date DESC);
CREATE INDEX IF NOT EXISTS idx_hotels_last_active ON hotels(last_active DESC);

-- Enable Row Level Security
ALTER TABLE hotels ENABLE ROW LEVEL SECURITY;

-- Policy: Allow service role full access
CREATE POLICY "allow_service_full_access_hotels" 
    ON hotels 
    FOR ALL 
    TO service_role
    USING (true)
    WITH CHECK (true);

-- Policy: Allow authenticated users to read their own hotel data
CREATE POLICY "allow_authenticated_read_own_hotel" 
    ON hotels 
    FOR SELECT 
    TO authenticated
    USING (hotel_id = current_setting('app.hotel_id', true));

-- Function to update updated_at timestamp
CREATE OR REPLACE FUNCTION update_hotels_updated_at()
RETURNS TRIGGER AS $$
BEGIN
    NEW.updated_at = TIMEZONE('utc'::text, NOW());
    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

-- Trigger to automatically update updated_at
CREATE TRIGGER hotels_updated_at_trigger
    BEFORE UPDATE ON hotels
    FOR EACH ROW
    EXECUTE FUNCTION update_hotels_updated_at();

-- Function to update last_active timestamp
CREATE OR REPLACE FUNCTION update_hotel_last_active(p_hotel_id VARCHAR)
RETURNS VOID AS $$
BEGIN
    UPDATE hotels 
    SET last_active = TIMEZONE('utc'::text, NOW())
    WHERE hotel_id = p_hotel_id;
END;
$$ LANGUAGE plpgsql;

-- Add comment
COMMENT ON TABLE hotels IS 'Stores all hotels connected to the SmartHotel SaaS platform';

-- Insert sample data for demonstration
INSERT INTO hotels (hotel_id, name, status, ai_configured, website, email, phone, rooms, subscription_plan, monthly_revenue) VALUES
('hotel_001', 'Luxury Beach Resort', 'active', true, 'https://luxurybeachresort.com', 'info@luxurybeachresort.com', '+1 555-0101', 150, 'premium', 5000),
('hotel_002', 'City Center Hotel', 'active', true, 'https://citycenterhotel.com', 'info@citycenterhotel.com', '+1 555-0102', 80, 'standard', 3000),
('hotel_003', 'Mountain View Lodge', 'active', false, 'https://mountainviewlodge.com', 'info@mountainviewlodge.com', '+1 555-0103', 45, 'standard', 2000),
('hotel_004', 'Grand Palace Hotel', 'pending', false, 'https://grandpalacehotel.com', 'info@grandpalacehotel.com', '+1 555-0104', 200, 'free', 0),
('hotel_005', 'Seaside Inn', 'active', true, 'https://seasideinn.com', 'info@seasideinn.com', '+1 555-0105', 30, 'standard', 1500)
ON CONFLICT (hotel_id) DO NOTHING;

-- Example queries:

-- Get all active hotels
-- SELECT * FROM hotels WHERE status = 'active' ORDER BY last_active DESC;

-- Get hotels with AI configured
-- SELECT * FROM hotels WHERE ai_configured = true;

-- Get total number of hotels by status
-- SELECT status, COUNT(*) as count FROM hotels GROUP BY status;

-- Update hotel last active time
-- SELECT update_hotel_last_active('hotel_001');