import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from "npm:@supabase/supabase-js@2";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers": "*",
};

interface MewsCredentials {
  hotelId: string;
  clientToken: string;
  accessToken: string;
  enterpriseId: string;
}

interface MigrationRequest {
  credentials: MewsCredentials;
  dataTypes: string[];
}

serve(async (req) => {
  const requestId = crypto.randomUUID();
  console.log(`[${requestId}] Mews migration request received`);

  if (req.method === "OPTIONS") {
    return new Response(null, { status: 204, headers: corsHeaders });
  }

  try {
    const supabaseUrl = Deno.env.get("SUPABASE_URL")!;
    const supabaseKey = Deno.env.get("SUPABASE_SERVICE_ROLE_KEY")!;
    const supabase = createClient(supabaseUrl, supabaseKey);

    const body = await req.json() as MigrationRequest;
    const { credentials, dataTypes } = body;

    console.log(`[${requestId}] Starting Mews migration for hotel: ${credentials.hotelId}`);

    const results = {
      guests: 0,
      bookings: 0,
      rooms: 0,
      errors: [] as string[],
    };

    const mewsApiUrl = "https://api.mews.com/api/connector/v1";

    // Migrate Guests
    if (dataTypes.includes("guests")) {
      console.log(`[${requestId}] Migrating guests from Mews...`);
      const guestsResult = await migrateGuestsFromMews(credentials, mewsApiUrl, supabase);
      results.guests = guestsResult.count;
      results.errors.push(...guestsResult.errors);
    }

    // Migrate Bookings
    if (dataTypes.includes("bookings")) {
      console.log(`[${requestId}] Migrating bookings from Mews...`);
      const bookingsResult = await migrateBookingsFromMews(credentials, mewsApiUrl, supabase);
      results.bookings = bookingsResult.count;
      results.errors.push(...bookingsResult.errors);
    }

    // Migrate Rooms
    if (dataTypes.includes("rooms")) {
      console.log(`[${requestId}] Migrating rooms from Mews...`);
      const roomsResult = await migrateRoomsFromMews(credentials, mewsApiUrl, supabase);
      results.rooms = roomsResult.count;
      results.errors.push(...roomsResult.errors);
    }

    console.log(`[${requestId}] Mews migration completed:`, results);

    return new Response(
      JSON.stringify({
        success: true,
        results,
        message: `Successfully migrated ${results.guests} guests, ${results.bookings} bookings, and ${results.rooms} rooms from Mews`,
      }),
      { headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );

  } catch (error) {
    console.error(`[${requestId}] Migration error:`, error);
    return new Response(
      JSON.stringify({ success: false, error: error.message }),
      { status: 400, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  }
});

async function migrateGuestsFromMews(credentials: MewsCredentials, apiUrl: string, supabase: any) {
  const errors: string[] = [];
  let count = 0;

  try {
    const response = await fetch(`${apiUrl}/customers/getAll`, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "ClientToken": credentials.clientToken,
        "AccessToken": credentials.accessToken,
      },
      body: JSON.stringify({
        EnterpriseIds: [credentials.enterpriseId],
      }),
    });

    if (!response.ok) {
      throw new Error(`Mews API error: ${response.statusText}`);
    }

    const data = await response.json();

    for (const customer of data.Customers || []) {
      try {
        const guestData = {
          hotel_id: credentials.hotelId,
          first_name: customer.FirstName || "",
          last_name: customer.LastName || "",
          email: customer.Email || "",
          phone: customer.Phone || "",
          address: customer.Address?.Line1 || "",
          city: customer.Address?.City || "",
          country: customer.Address?.CountryCode || "",
          external_id: customer.Id,
          created_at: new Date().toISOString(),
        };

        const { error } = await supabase
          .from(`app_${credentials.hotelId}_guests`)
          .insert(guestData);

        if (error) {
          errors.push(`Guest ${customer.FirstName} ${customer.LastName}: ${error.message}`);
        } else {
          count++;
        }
      } catch (err) {
        errors.push(`Guest processing error: ${err.message}`);
      }
    }

  } catch (error) {
    errors.push(`Mews guests migration error: ${error.message}`);
  }

  return { count, errors };
}

async function migrateBookingsFromMews(credentials: MewsCredentials, apiUrl: string, supabase: any) {
  const errors: string[] = [];
  let count = 0;

  try {
    const response = await fetch(`${apiUrl}/reservations/getAll`, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "ClientToken": credentials.clientToken,
        "AccessToken": credentials.accessToken,
      },
      body: JSON.stringify({
        EnterpriseIds: [credentials.enterpriseId],
        StartUtc: new Date(Date.now() - 365 * 24 * 60 * 60 * 1000).toISOString(), // Last year
        EndUtc: new Date().toISOString(),
      }),
    });

    if (!response.ok) {
      throw new Error(`Mews API error: ${response.statusText}`);
    }

    const data = await response.json();

    for (const reservation of data.Reservations || []) {
      try {
        const bookingData = {
          hotel_id: credentials.hotelId,
          guest_name: `${reservation.Customer?.FirstName || ""} ${reservation.Customer?.LastName || ""}`,
          guest_email: reservation.Customer?.Email || "",
          guest_phone: reservation.Customer?.Phone || "",
          room_type: reservation.RoomCategory?.Name || "",
          check_in: reservation.StartUtc,
          check_out: reservation.EndUtc,
          adults: reservation.AdultCount || 1,
          children: reservation.ChildCount || 0,
          total_amount: reservation.TotalAmount?.Value || 0,
          status: reservation.State || "confirmed",
          external_id: reservation.Id,
          created_at: new Date().toISOString(),
        };

        const { error } = await supabase
          .from(`app_${credentials.hotelId}_bookings`)
          .insert(bookingData);

        if (error) {
          errors.push(`Booking ${reservation.Id}: ${error.message}`);
        } else {
          count++;
        }
      } catch (err) {
        errors.push(`Booking processing error: ${err.message}`);
      }
    }

  } catch (error) {
    errors.push(`Mews bookings migration error: ${error.message}`);
  }

  return { count, errors };
}

async function migrateRoomsFromMews(credentials: MewsCredentials, apiUrl: string, supabase: any) {
  const errors: string[] = [];
  let count = 0;

  try {
    const response = await fetch(`${apiUrl}/resources/getAll`, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "ClientToken": credentials.clientToken,
        "AccessToken": credentials.accessToken,
      },
      body: JSON.stringify({
        EnterpriseIds: [credentials.enterpriseId],
      }),
    });

    if (!response.ok) {
      throw new Error(`Mews API error: ${response.statusText}`);
    }

    const data = await response.json();

    for (const resource of data.Resources || []) {
      try {
        const roomData = {
          hotel_id: credentials.hotelId,
          room_number: resource.Name,
          room_type: resource.Type || "standard",
          floor: resource.Floor || 1,
          capacity: resource.Capacity || 2,
          status: resource.State === "Dirty" ? "cleaning" : "available",
          external_id: resource.Id,
          created_at: new Date().toISOString(),
        };

        const { error } = await supabase
          .from(`app_${credentials.hotelId}_rooms`)
          .insert(roomData);

        if (error) {
          errors.push(`Room ${resource.Name}: ${error.message}`);
        } else {
          count++;
        }
      } catch (err) {
        errors.push(`Room processing error: ${err.message}`);
      }
    }

  } catch (error) {
    errors.push(`Mews rooms migration error: ${error.message}`);
  }

  return { count, errors };
}