import { serve } from 'https://deno.land/std@0.168.0/http/server.ts'
import { createClient } from 'https://esm.sh/@supabase/supabase-js@2'

const corsHeaders = {
  'Access-Control-Allow-Origin': '*',
  'Access-Control-Allow-Headers': 'authorization, x-client-info, apikey, content-type',
}

interface ConversationMessage {
  text: string
  sender: string
  time: string
  attachment?: any
}

interface AIRequest {
  message: string
  attachment?: any
  history: ConversationMessage[]
  hotelId: string
  sessionId?: string
}

serve(async (req) => {
  // Generate unique request ID for logging
  const requestId = crypto.randomUUID()
  console.log(`[${requestId}] New AI Concierge request`)

  // Handle CORS preflight
  if (req.method === 'OPTIONS') {
    return new Response('ok', { headers: corsHeaders })
  }

  try {
    const { message, attachment, history, hotelId, sessionId }: AIRequest = await req.json()
    console.log(`[${requestId}] Hotel: ${hotelId}, Session: ${sessionId}, Message: ${message}`)
    
    // Initialize Supabase client
    const supabaseUrl = Deno.env.get('SUPABASE_URL')!
    const supabaseKey = Deno.env.get('SUPABASE_SERVICE_ROLE_KEY')!
    const supabase = createClient(supabaseUrl, supabaseKey)
    
    // Get or create session ID
    const currentSessionId = sessionId || crypto.randomUUID()
    
    // Get hotel-specific OpenAI API key
    const openaiKey = Deno.env.get(`APP_${hotelId.toUpperCase()}_OPENAI_API_KEY`)
    
    if (!openaiKey) {
      console.error(`[${requestId}] OpenAI API key not configured for hotel ${hotelId}`)
      throw new Error('AI service not configured for this hotel')
    }
    
    console.log(`[${requestId}] OpenAI key found for hotel ${hotelId}`)
    
    // Store user message in database
    await supabase
      .from(`app_${hotelId}_ai_messages`)
      .insert({
        session_id: currentSessionId,
        message: message,
        sender: 'user',
        has_attachment: !!attachment,
        attachment_info: attachment ? JSON.stringify(attachment) : null,
        created_at: new Date().toISOString()
      })
    
    console.log(`[${requestId}] User message stored in database`)
    
    // Retrieve full conversation history from database
    const { data: dbHistory } = await supabase
      .from(`app_${hotelId}_ai_messages`)
      .select('*')
      .eq('session_id', currentSessionId)
      .order('created_at', { ascending: true })
      .limit(20) // Last 20 messages for context
    
    console.log(`[${requestId}] Retrieved ${dbHistory?.length || 0} messages from database`)
    
    // Fetch hotel data for context
    const hotelContext = await getHotelContext(supabase, hotelId, requestId)
    console.log(`[${requestId}] Hotel context loaded: ${hotelContext.rooms.length} rooms, ${hotelContext.menuItems.length} menu items`)
    
    // Build conversation history for OpenAI using database history
    const conversationHistory = dbHistory || []
    const messages = [
      {
        role: 'system',
        content: buildSystemPrompt(hotelContext, currentSessionId)
      },
      ...conversationHistory.map(msg => ({
        role: msg.sender === 'user' ? 'user' : 'assistant',
        content: msg.message
      }))
    ]
    
    console.log(`[${requestId}] Calling OpenAI API with ${messages.length} messages...`)
    
    // Call OpenAI API with hotel-specific key
    const openaiResponse = await fetch('https://api.openai.com/v1/chat/completions', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${openaiKey}`
      },
      body: JSON.stringify({
        model: 'gpt-4-turbo-preview',
        messages,
        max_tokens: 600,
        temperature: 0.7,
        functions: getAvailableFunctions(),
        function_call: 'auto'
      })
    })
    
    if (!openaiResponse.ok) {
      const errorText = await openaiResponse.text()
      console.error(`[${requestId}] OpenAI API error: ${openaiResponse.status} - ${errorText}`)
      throw new Error('OpenAI API error')
    }
    
    const aiData = await openaiResponse.json()
    const aiMessage = aiData.choices[0].message
    
    console.log(`[${requestId}] OpenAI response received`)
    
    // Check if AI wants to call a function
    let responseText = aiMessage.content
    let actions = []
    
    if (aiMessage.function_call) {
      console.log(`[${requestId}] Function call: ${aiMessage.function_call.name}`)
      const functionResult = await handleFunctionCall(
        aiMessage.function_call.name,
        JSON.parse(aiMessage.function_call.arguments),
        supabase,
        hotelId,
        requestId
      )
      
      responseText = functionResult.text
      actions = functionResult.actions || []
    }
    
    // Store AI response in database
    await supabase
      .from(`app_${hotelId}_ai_messages`)
      .insert({
        session_id: currentSessionId,
        message: responseText,
        sender: 'assistant',
        has_attachment: false,
        function_called: aiMessage.function_call?.name || null,
        function_args: aiMessage.function_call ? JSON.stringify(aiMessage.function_call.arguments) : null,
        actions: actions.length > 0 ? JSON.stringify(actions) : null,
        created_at: new Date().toISOString()
      })
    
    console.log(`[${requestId}] AI response stored in database`)
    
    // Also log in conversations table for analytics
    await supabase
      .from(`app_${hotelId}_ai_conversations`)
      .insert({
        session_id: currentSessionId,
        user_message: message,
        ai_response: responseText,
        has_attachment: !!attachment,
        function_called: aiMessage.function_call?.name || null,
        created_at: new Date().toISOString()
      })
    
    console.log(`[${requestId}] Conversation logged successfully`)
    
    return new Response(
      JSON.stringify({
        text: responseText,
        actions,
        speak: true,
        sessionId: currentSessionId
      }),
      {
        headers: { ...corsHeaders, 'Content-Type': 'application/json' }
      }
    )
    
  } catch (error) {
    console.error(`[${requestId}] AI Concierge Error:`, error)
    return new Response(
      JSON.stringify({
        text: "I apologize, but I'm experiencing technical difficulties. Please contact our front desk for immediate assistance at the number provided on our website.",
        actions: [
          {
            type: 'contact_staff',
            label: 'Contact Front Desk',
            params: {}
          }
        ],
        speak: false
      }),
      {
        status: 500,
        headers: { ...corsHeaders, 'Content-Type': 'application/json' }
      }
    )
  }
})

async function getHotelContext(supabase: any, hotelId: string, requestId: string) {
  console.log(`[${requestId}] Fetching hotel context...`)
  
  // Fetch rooms
  const { data: rooms } = await supabase
    .from(`app_${hotelId}_rooms`)
    .select('*')
    .eq('available', true)
  
  // Fetch menu items
  const { data: menuItems } = await supabase
    .from(`app_${hotelId}_menu_items`)
    .select('*')
    .eq('available', true)
  
  // Fetch amenities
  const { data: amenities } = await supabase
    .from(`app_${hotelId}_amenities`)
    .select('*')
  
  return {
    rooms: rooms || [],
    menuItems: menuItems || [],
    amenities: amenities || [],
    hotelName: 'Luxury Hotel',
    checkInTime: '15:00',
    checkOutTime: '11:00',
    address: '123 Luxury Avenue, Athens, 10563, Greece',
    phone: '+30 210 1234567',
    email: 'info@luxuryhotel.com'
  }
}

function buildSystemPrompt(context: any, sessionId: string) {
  return `You are an AI concierge for ${context.hotelName}, a luxury hotel. You have access to real-time hotel data and can assist guests with:

SESSION CONTEXT:
- You are maintaining a continuous conversation with this guest
- Session ID: ${sessionId}
- Remember previous messages and context from this conversation
- Reference earlier parts of the conversation when relevant
- Build upon previous interactions naturally

1. ROOM BOOKINGS
Available rooms:
${context.rooms.map((r: any) => `- ${r.name} (${r.room_type}): €${r.price_per_night}/night, max ${r.max_occupancy} guests, ${r.size_sqm}m²`).join('\n')}

2. RESTAURANT & DINING
Menu highlights:
${context.menuItems.slice(0, 10).map((m: any) => `- ${m.name} (${m.category}): €${m.price}`).join('\n')}

Dining hours:
- Breakfast: 7:00 AM - 11:00 AM
- Lunch: 12:00 PM - 3:00 PM
- Dinner: 6:00 PM - 11:00 PM

3. AMENITIES
${context.amenities.map((a: any) => `- ${a.name}: ${a.description}`).join('\n')}

4. CHECK-IN/CHECK-OUT
- Check-in: From ${context.checkInTime}
- Check-out: Until ${context.checkOutTime}
- Self-service check-in available

5. HOTEL INFORMATION
- Address: ${context.address}
- Phone: ${context.phone}
- Email: ${context.email}

CONVERSATION GUIDELINES:
- Maintain context throughout the conversation - remember what the guest has told you
- Reference previous messages naturally (e.g., "As you mentioned earlier...")
- Build on previous interactions (e.g., if they asked about rooms, later suggest booking)
- Be friendly, professional, and conversational
- Provide specific information from the data above
- Offer to perform actions (book rooms, make reservations, etc.)
- Ask clarifying questions when needed
- Use emojis appropriately to enhance communication
- If you don't have information, offer to connect them with staff
- Always provide helpful next steps or actions
- Track the guest's preferences and needs throughout the conversation

MEMORY INSTRUCTIONS:
- You have access to the full conversation history
- Always consider previous messages when responding
- If the guest refers to something they said earlier, acknowledge it
- Maintain continuity in the conversation
- Remember guest preferences (dates, room types, dietary needs, etc.)

When appropriate, use function calls to:
- Check room availability
- Make restaurant reservations
- Provide booking links
- Show amenity details`
}

function getAvailableFunctions() {
  return [
    {
      name: 'check_room_availability',
      description: 'Check if rooms are available for specific dates',
      parameters: {
        type: 'object',
        properties: {
          checkin: { type: 'string', description: 'Check-in date (YYYY-MM-DD)' },
          checkout: { type: 'string', description: 'Check-out date (YYYY-MM-DD)' },
          guests: { type: 'number', description: 'Number of guests' }
        },
        required: ['checkin', 'checkout']
      }
    },
    {
      name: 'get_room_details',
      description: 'Get detailed information about a specific room',
      parameters: {
        type: 'object',
        properties: {
          room_id: { type: 'string', description: 'Room ID' }
        },
        required: ['room_id']
      }
    },
    {
      name: 'make_restaurant_reservation',
      description: 'Create a restaurant reservation',
      parameters: {
        type: 'object',
        properties: {
          date: { type: 'string', description: 'Reservation date (YYYY-MM-DD)' },
          time: { type: 'string', description: 'Reservation time (HH:MM)' },
          guests: { type: 'number', description: 'Number of guests' }
        },
        required: ['date', 'time', 'guests']
      }
    },
    {
      name: 'get_local_recommendations',
      description: 'Get recommendations for local attractions, restaurants, or activities',
      parameters: {
        type: 'object',
        properties: {
          category: { 
            type: 'string', 
            enum: ['restaurants', 'attractions', 'shopping', 'nightlife', 'museums'],
            description: 'Category of recommendations'
          }
        },
        required: ['category']
      }
    }
  ]
}

async function handleFunctionCall(functionName: string, args: any, supabase: any, hotelId: string, requestId: string) {
  console.log(`[${requestId}] Executing function: ${functionName}`)
  
  switch (functionName) {
    case 'check_room_availability':
      return await checkRoomAvailability(args, supabase, hotelId)
    
    case 'get_room_details':
      return await getRoomDetails(args, supabase, hotelId)
    
    case 'make_restaurant_reservation':
      return await makeRestaurantReservation(args, supabase, hotelId)
    
    case 'get_local_recommendations':
      return await getLocalRecommendations(args)
    
    default:
      return {
        text: "I can help you with that. Let me know what you'd like to do next.",
        actions: []
      }
  }
}

async function checkRoomAvailability(args: any, supabase: any, hotelId: string) {
  const { checkin, checkout, guests = 2 } = args
  
  const { data: rooms } = await supabase
    .from(`app_${hotelId}_rooms`)
    .select('*')
    .eq('available', true)
    .gte('max_occupancy', guests)
  
  if (!rooms || rooms.length === 0) {
    return {
      text: `I'm sorry, but we don't have any rooms available for ${guests} guests from ${checkin} to ${checkout}. Would you like to try different dates or contact our reservations team?`,
      actions: [
        {
          type: 'contact_staff',
          label: 'Contact Reservations',
          params: {}
        }
      ]
    }
  }
  
  const roomList = rooms.map(r => 
    `• ${r.name} - €${r.price_per_night}/night (${r.room_type}, ${r.max_occupancy} guests)`
  ).join('\n')
  
  return {
    text: `Great news! We have ${rooms.length} room(s) available for ${guests} guests from ${checkin} to ${checkout}:\n\n${roomList}\n\nWould you like to book one of these rooms?`,
    actions: rooms.slice(0, 3).map((r: any) => ({
      type: 'book_room',
      label: `Book ${r.name}`,
      params: { room_id: r.id, checkin, checkout }
    }))
  }
}

async function getRoomDetails(args: any, supabase: any, hotelId: string) {
  const { room_id } = args
  
  const { data: room } = await supabase
    .from(`app_${hotelId}_rooms`)
    .select('*')
    .eq('id', room_id)
    .single()
  
  if (!room) {
    return {
      text: "I couldn't find that room. Would you like to see all available rooms?",
      actions: [
        {
          type: 'check_availability',
          label: 'View All Rooms',
          params: {}
        }
      ]
    }
  }
  
  return {
    text: `**${room.name}**\n\n${room.description}\n\n• Type: ${room.room_type}\n• Price: €${room.price_per_night}/night\n• Capacity: ${room.max_occupancy} guests\n• Size: ${room.size_sqm}m²\n• Amenities: ${room.amenities}\n\nWould you like to book this room?`,
    actions: [
      {
        type: 'book_room',
        label: 'Book This Room',
        params: { room_id: room.id }
      }
    ]
  }
}

async function makeRestaurantReservation(args: any, supabase: any, hotelId: string) {
  const { date, time, guests } = args
  
  return {
    text: `Perfect! I can reserve a table for ${guests} guests on ${date} at ${time}.\n\nTo confirm your reservation, please click the button below. You'll receive a confirmation email once it's confirmed.`,
    actions: [
      {
        type: 'make_reservation',
        label: 'Confirm Reservation',
        params: { date, time, guests }
      }
    ]
  }
}

async function getLocalRecommendations(args: any) {
  const { category } = args
  
  const recommendations: Record<string, string> = {
    restaurants: `Here are some excellent local restaurants near our hotel:

🍽️ **Varoulko Seaside** (5 min walk)
- Michelin-starred seafood restaurant
- Stunning harbor views
- €€€€

🍕 **Funky Gourmet** (10 min walk)
- Modern Greek cuisine
- 2 Michelin stars
- €€€€

🥘 **Ta Karamanlidika tou Fani** (15 min walk)
- Traditional Greek deli
- Authentic local experience
- €€

Would you like directions or help making a reservation?`,
    
    attractions: `Top attractions near our hotel:

🏛️ **Acropolis** (20 min walk)
- Ancient citadel with Parthenon
- Must-see historical site
- Open 8 AM - 8 PM

🏛️ **Acropolis Museum** (25 min walk)
- World-class archaeological museum
- Modern architecture
- Open 9 AM - 5 PM

🌊 **Plaka District** (15 min walk)
- Historic neighborhood
- Shopping and dining
- Perfect for evening strolls

Need transportation or tour booking assistance?`,
    
    shopping: `Best shopping areas nearby:

🛍️ **Ermou Street** (10 min walk)
- Main shopping street
- International brands
- Pedestrian-friendly

👜 **Kolonaki** (15 min walk)
- Upscale boutiques
- Designer stores
- Trendy cafes

🎨 **Monastiraki Flea Market** (20 min walk)
- Antiques and souvenirs
- Local crafts
- Open daily

Would you like directions or shopping tips?`,
    
    nightlife: `Popular nightlife spots:

🍸 **Brettos Bar** (15 min walk)
- Oldest distillery in Athens
- Colorful atmosphere
- Local spirits

🎵 **Gazarte** (20 min walk)
- Rooftop bar
- Live music
- Great city views

💃 **Lohan Nightclub** (25 min walk)
- Upscale nightclub
- International DJs
- Dress code applies

Need transportation or VIP table reservations?`,
    
    museums: `Must-visit museums:

🎨 **National Archaeological Museum** (15 min walk)
- Largest archaeological museum in Greece
- Ancient Greek art and artifacts
- Open 8 AM - 8 PM

🖼️ **Benaki Museum** (20 min walk)
- Greek culture and history
- Beautiful collections
- Open 9 AM - 5 PM

🏛️ **Byzantine Museum** (25 min walk)
- Byzantine and post-Byzantine art
- Religious artifacts
- Open 8 AM - 8 PM

Would you like tickets or guided tour information?`
  }
  
  return {
    text: recommendations[category] || "I can provide recommendations for restaurants, attractions, shopping, nightlife, or museums. What interests you?",
    actions: []
  }
}